def create_directories(preprocessing, spectral, inversion, swi_dir) -> dict[str, list]:
    """
    Creates directory structure for SWI (Seismic Wave Inversion) processing workflow.

    Constructs organized directory tree for storing preprocessing, spectral analysis,
    inversion, and postprocessing results based on data type and processing parameters.

    Args:
        preprocessing: Preprocessing configuration object containing data directory
                      and type information.
        spectral: Spectral analysis configuration object with method selection.
        inversion: Inversion configuration object with method selection.
        swi_dir: Root directory where all SWI processing directories will be created.

    Returns:
        dict[str, list]: Dictionary mapping module names to lists of created directories.
        Keys: 'preprocessing', 'spectral_analysis', 'inversion', 'postprocessing'
    """
    # Determine root directory name from data path
    data_dir = preprocessing.data_dir
    if data_dir.is_dir():
        folder_name = data_dir.name
    else:
        folder_name = data_dir.parent.name

    # Determine data type and create appropriate layout structure
    if (type_data := preprocessing.type_data.value) == "2d":
        save_layout = [f"offsets_{preprocessing.offset_min}_{preprocessing.offset_max}"]
    elif preprocessing.parameters_3d.sort_3d_order == 'csp':
        save_layout = [f"num_sectors_{preprocessing.parameters_3d.num_sectors}"]
    else:
        save_layout = ['cdp']

    # Define directory structure templates for each processing module
    base_dirs = [
        # Preprocessing directories
        (f"{type_data}/preprocessing", save_layout),

        # Spectral analysis directories (dc, image, segy subfolders)
        (
            f"{type_data}/spectral_analysis",
            [f"{sl}/{spectral.spectral_method.value}/{subdir}" for
             sl in save_layout for subdir in ["dc", "image", "segy"]],
        ),

        # Inversion directories (models1d_bin, image subfolders)
        (
            f"{type_data}/inversion",
            [f"{sl}/{spectral.spectral_method.value}/{inversion.inversion_method.value}/{subdir}" for
             sl in save_layout for subdir in ["models1d_bin", "image"]],
        ),

        # Postprocessing directories (models2d_bin, image, segy, fdm subfolders)
        (
            f"{type_data}/postprocessing",
            [f"{sl}/{spectral.spectral_method.value}/{inversion.inversion_method.value}/{subdir}" for
             sl in save_layout for subdir in ["models2d_bin", "image", "segy", "fdm"]],
        ),
    ]

    # Create directories and track them by module
    module_dirs = {
        "preprocessing": [],
        "spectral_analysis": [],
        "inversion": [],
        "postprocessing": []
    }

    for base, subdirs in base_dirs:
        for subdir in subdirs:
            # Construct full directory path
            directory = swi_dir / base / folder_name / subdir
            # Create directory with parents (if needed)
            directory.mkdir(parents=True, exist_ok=True)
            # Extract module name from base path and add directory to list
            module_name = base.split('/')[1]  # Extract 'preprocessing', 'spectral_analysis', etc.
            module_dirs[module_name].append(directory)

    return module_dirs